/**
 * InstantPay admin iframe helper.
 *
 * Adjusts the height of the embedded InstantPay dashboard in wp-admin
 * and validates postMessage origin.
 */

// Accordion toggle for "How to use" section - works independently
(function initHowToAccordion() {
  if (typeof window === 'undefined') return;
  
  function setupAccordion() {
    var howtoToggle = document.getElementById('instantpay-howto-toggle');
    var howtoContent = document.getElementById('instantpay-howto-content');
    if (howtoToggle && howtoContent) {
      // Initially hide content
      howtoContent.classList.remove('instantpay-howto-content--open');
      
      howtoToggle.addEventListener('click', function (e) {
        e.preventDefault();
        e.stopPropagation();
        var isExpanded = howtoToggle.getAttribute('aria-expanded') === 'true';
        var newExpanded = !isExpanded;
        howtoToggle.setAttribute('aria-expanded', newExpanded);
        if (newExpanded) {
          howtoContent.classList.add('instantpay-howto-content--open');
        } else {
          howtoContent.classList.remove('instantpay-howto-content--open');
        }
      });
    }
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', setupAccordion);
  } else {
    setupAccordion();
  }
})();

// Main admin iframe functionality
(function () {
  if (typeof window === 'undefined') return;
  if (typeof window.InstantPayAdmin === 'undefined') return;

  var config = window.InstantPayAdmin;
  var iframeId = config.iframeId || 'instantpay-admin-iframe';
  // Varsayılan olarak production domain'i www ile kullan.
  var allowedOrigin = config.allowedOrigin || 'https://www.instantpay.now';
  var ajaxUrl = config.ajaxUrl || (typeof window.ajaxurl === 'string' ? window.ajaxurl : null);

  function isAllowedOrigin(origin) {
    return origin === allowedOrigin;
  }

  function getIframe() {
    return document.getElementById(iframeId);
  }

  function resizeIframe(height) {
    var iframe = getIframe();
    if (!iframe || typeof height !== 'number') return;
    iframe.style.height = height + 'px';
  }

  // Refresh status butonu: iframe'den mevcut durumu sorgula, sonra sayfayı yenile.
  document.addEventListener('DOMContentLoaded', function () {
    var refreshBtn = document.getElementById('instantpay-refresh-button');
    var iframe = getIframe();
    
    if (refreshBtn && iframe) {
      refreshBtn.addEventListener('click', function (event) {
        event.preventDefault();
        
        // Iframe'e mevcut durumu sorgulayan bir mesaj gönder
        // Iframe bu mesajı alınca postMessage ile cevap verecek
        if (iframe.contentWindow) {
          try {
            iframe.contentWindow.postMessage(
              {
                type: 'instantpay-admin-request-status'
              },
              allowedOrigin
            );
          } catch (e) {
            // Cross-origin hatası olabilir, sessizce geç
            console.warn('Could not send status request to iframe:', e);
          }
        }
        
        // Kısa bir gecikme sonrası sayfayı yenile (iframe'in cevap vermesi için zaman tanı)
        setTimeout(function () {
          window.location.reload();
        }, 500);
      });
    }
  });

  window.addEventListener(
    'message',
    function (event) {
      if (!isAllowedOrigin(event.origin)) return;

      var data = event.data || {};

      // Generic { height } payload.
      if (typeof data.height === 'number') {
        resizeIframe(data.height);
        return;
      }

      // Namespaced event support, e.g. { type: 'instantpay-admin-resize', payload: { height } }.
      if (data.type === 'instantpay-admin-resize' && data.payload && typeof data.payload.height === 'number') {
        resizeIframe(data.payload.height);
        return;
      }

      // Connection update: username + site_url (+ optional token) from embedded dashboard.
      if (data.type === 'instantpay-admin-update-connection' && data.payload && ajaxUrl && window.jQuery) {
        var payload = data.payload;
        window.jQuery.post(ajaxUrl, {
          action: 'instantpay_update_connection',
          nonce: config.nonce || '',
          username: payload.username || '',
          site_url: payload.siteUrl || '',
          site_token: payload.siteToken || ''
        });
        return;
      }

      // Connection clear: kullanıcı çıkış yaptı veya bağlantı koptu.
      if (data.type === 'instantpay-admin-clear-connection' && data.payload && ajaxUrl && window.jQuery) {
        var clearPayload = data.payload;
        // WordPress'te username'i tamamen kaldırmak için özel bir değer gönderiyoruz
        // AJAX handler'da bu değeri null olarak işleyeceğiz
        window.jQuery.post(ajaxUrl, {
          action: 'instantpay_update_connection',
          nonce: config.nonce || '',
          username: '__CLEAR__', // Özel değer: WordPress'te null yapılacak
          site_url: clearPayload.siteUrl || '',
          site_token: ''
        });
      }
    },
    false
  );
})();


