/**
 * InstantPay Gutenberg blocks editor script.
 *
 * Provides Inspector controls and editor-side UX for:
 * - instantpay/donate
 * - instantpay/product
 * - instantpay/page
 */

(function () {
  if (typeof window === 'undefined' || typeof window.wp === 'undefined') return;

  var wp = window.wp;
  var registerBlockType = wp.blocks && wp.blocks.registerBlockType;
  var InspectorControls = (wp.blockEditor || wp.editor).InspectorControls;
  var PanelBody = wp.components.PanelBody;
  var TextControl = wp.components.TextControl;
  var SelectControl = wp.components.SelectControl;
  var __ = wp.i18n.__;
  var useState = wp.element.useState;
  var useEffect = wp.element.useEffect;

  if (!registerBlockType || !InspectorControls) {
    return;
  }

  /**
   * Common theme options.
   */
  var themeOptions = [
    { label: __('Auto (use widget default)', 'instantpay'), value: '' },
    { label: __('Light', 'instantpay'), value: 'light' },
    { label: __('Dark', 'instantpay'), value: 'dark' },
  ];

  /**
   * Donate block.
   */
  registerBlockType('instantpay/donate', {
    title: __('InstantPay Donate Button', 'instantpay'),
    description: __(
      'Add a Solana donation button powered by your InstantPay account.',
      'instantpay'
    ),
    icon: 'heart',
    category: 'widgets',
    supports: {
      align: true,
    },
    edit: function DonateEdit(props) {
      var attributes = props.attributes;

      return (
        wp.element.createElement(
          wp.element.Fragment,
          null,
          wp.element.createElement(
            InspectorControls,
            null,
            wp.element.createElement(
              PanelBody,
              {
                title: __('Donate Settings', 'instantpay'),
                initialOpen: true,
              },
              wp.element.createElement(TextControl, {
                label: __('Default amount (optional, in SOL)', 'instantpay'),
                help: __(
                  'If you leave this empty, visitors can choose how much to donate.',
                  'instantpay'
                ),
                placeholder: 'e.g. 0.5',
                value: attributes.amount || '',
                onChange: function (value) {
                  props.setAttributes({ amount: value });
                },
              }),
              wp.element.createElement(SelectControl, {
                label: __('Theme', 'instantpay'),
                value: attributes.theme || '',
                options: themeOptions,
                onChange: function (value) {
                  props.setAttributes({ theme: value });
                },
              })
            )
          ),
          wp.element.createElement(
            'div',
            {
              className:
                'instantpay-block-preview instantpay-block-preview-donate',
              style: {
                border: '1px dashed #e5e7eb',
                padding: '12px 14px',
                borderRadius: '8px',
                background: '#ffffff',
              },
            },
            wp.element.createElement(
              'strong',
              null,
              __('InstantPay Donate Button', 'instantpay')
            ),
            wp.element.createElement('br', null),
            wp.element.createElement(
              'span',
              { style: { fontSize: '12px', color: '#6b7280' } },
              attributes.amount
                ? __('Default amount:', 'instantpay') +
                    ' ' +
                    attributes.amount +
                    ' SOL'
                : __('Visitors can choose the amount', 'instantpay')
            )
          )
        )
      );
    },
    save: function () {
      // Dynamic block – rendered via PHP shortcode on the frontend.
      return null;
    },
  });

  /**
   * Product block.
   */
  registerBlockType('instantpay/product', {
    title: __('InstantPay Product Purchase', 'instantpay'),
    description: __(
      'Add a digital product purchase widget powered by InstantPay.',
      'instantpay'
    ),
    icon: 'cart',
    category: 'widgets',
    supports: {
      align: true,
    },
    edit: function ProductEdit(props) {
      var attributes = props.attributes;

      var _useState = useState({
          products: null,
          isLoading: false,
          error: null,
          useManual: false,
        }),
        state = _useState[0],
        setState = _useState[1];

      useEffect(
        function () {
          // Eğer global konfig yoksa veya username yoksa fetch denemeyelim.
          if (
            typeof InstantPayBlocks === 'undefined' ||
            !InstantPayBlocks.apiBase ||
            !InstantPayBlocks.username
          ) {
            return;
          }

          setState(function (prev) {
            return Object.assign({}, prev, {
              isLoading: true,
              error: null,
            });
          });

          var url =
            InstantPayBlocks.apiBase +
            '/api/public/products-by-username?username=' +
            encodeURIComponent(InstantPayBlocks.username);

          window
            .fetch(url)
            .then(function (res) {
              if (!res.ok) {
                throw new Error('Request failed: ' + res.status);
              }
              return res.json();
            })
            .then(function (data) {
              setState(function (prev) {
                return Object.assign({}, prev, {
                  products: Array.isArray(data.products)
                    ? data.products
                    : [],
                  isLoading: false,
                  error: null,
                });
              });
            })
            .catch(function () {
              setState(function (prev) {
                return Object.assign({}, prev, {
                  isLoading: false,
                  error: 'failed',
                });
              });
            });
        },
        [] // sadece ilk mount'ta çalışsın
      );

      var hasConfig =
        typeof InstantPayBlocks !== 'undefined' &&
        !!InstantPayBlocks.username;

      var hasProducts =
        state.products && state.products.length && !state.error;

      var showManualInput =
        !hasProducts || state.useManual || state.error || !hasConfig;

      return wp.element.createElement(
        wp.element.Fragment,
        null,
        wp.element.createElement(
          InspectorControls,
          null,
          wp.element.createElement(
            PanelBody,
            {
              title: __('Product Settings', 'instantpay'),
              initialOpen: true,
            },
            !hasConfig &&
              wp.element.createElement(
                'p',
                { style: { fontSize: '12px', color: '#b91c1c' } },
                __(
                  'InstantPay account is not connected yet. Please open the InstantPay admin page and connect your wallet.',
                  'instantpay'
                )
              ),
            state.isLoading &&
              wp.element.createElement(
                'p',
                { style: { fontSize: '12px', color: '#6b7280' } },
                __('Loading your products from InstantPay…', 'instantpay')
              ),
            state.error &&
              wp.element.createElement(
                'p',
                { style: { fontSize: '12px', color: '#b91c1c' } },
                __(
                  'We could not load your products automatically. For now, please paste a Product ID manually.',
                  'instantpay'
                )
              ),
            hasProducts &&
              !state.useManual &&
              wp.element.createElement(SelectControl, {
                label: __('Choose a product', 'instantpay'),
                help: __(
                  'Select one of your existing digital products from InstantPay.',
                  'instantpay'
                ),
                value: attributes.id || '',
                options: [
                  {
                    label: __('Select a product…', 'instantpay'),
                    value: '',
                  },
                ].concat(
                  state.products.map(function (p) {
                    var label = p.title || p.id;
                    if (p.price_sol) {
                      label += ' – ' + p.price_sol + ' SOL';
                    }
                    return {
                      label: label,
                      value: p.id,
                    };
                  })
                ),
                onChange: function (value) {
                  props.setAttributes({ id: value });
                },
              }),
            hasProducts &&
              wp.element.createElement(
                'button',
                {
                  type: 'button',
                  className: 'components-button is-link is-small',
                  onClick: function () {
                    setState(function (prev) {
                      return Object.assign({}, prev, {
                        useManual: !prev.useManual,
                      });
                    });
                  },
                },
                state.useManual
                  ? __('Use product dropdown instead', 'instantpay')
                  : __('Enter Product ID manually', 'instantpay')
              ),
            !state.isLoading &&
              state.products &&
              state.products.length === 0 &&
              wp.element.createElement(
                'p',
                { style: { fontSize: '12px', color: '#6b7280' } },
                __(
                  'You do not have any digital products yet. Create your first product in the InstantPay dashboard.',
                  'instantpay'
                ),
                ' ',
                wp.element.createElement(
                  'a',
                  {
                    href:
                      'https://www.instantpay.now/dashboard/products',
                    target: '_blank',
                    rel: 'noopener noreferrer',
                  },
                  __('Open InstantPay Products', 'instantpay')
                )
              ),
            showManualInput &&
              wp.element.createElement(TextControl, {
                label: __('Product ID', 'instantpay'),
                help: __(
                  'If needed, you can paste a Product ID from your InstantPay dashboard.',
                  'instantpay'
                ),
                placeholder: 'prod_123',
                value: attributes.id || '',
                onChange: function (value) {
                  props.setAttributes({ id: value });
                },
              }),
            wp.element.createElement(SelectControl, {
              label: __('Theme', 'instantpay'),
              value: attributes.theme || '',
              options: themeOptions,
              onChange: function (value) {
                props.setAttributes({ theme: value });
              },
            })
          )
        ),
        wp.element.createElement(
          'div',
          {
            className:
              'instantpay-block-preview instantpay-block-preview-product',
            style: {
              border: '1px dashed #e5e7eb',
              padding: '12px 14px',
              borderRadius: '8px',
              background: '#ffffff',
            },
          },
          wp.element.createElement(
            'strong',
            null,
            __('InstantPay Product Widget', 'instantpay')
          ),
          wp.element.createElement('br', null),
          wp.element.createElement(
            'span',
            { style: { fontSize: '12px', color: '#6b7280' } },
            attributes.id
              ? __('Product ID:', 'instantpay') + ' ' + attributes.id
              : hasProducts
              ? __('Select a product from the sidebar', 'instantpay')
              : __('No product selected yet', 'instantpay')
          )
        )
      );
    },
    save: function () {
      // Dynamic block – rendered via PHP shortcode on the frontend.
      return null;
    },
  });

  /**
   * Full payment page block.
   */
  registerBlockType('instantpay/page', {
    title: __('InstantPay Full Page Embed', 'instantpay'),
    description: __(
      'Embed your full InstantPay payment page inside this post or page.',
      'instantpay'
    ),
    icon: 'admin-site-alt3',
    category: 'widgets',
    supports: {
      align: true,
    },
    edit: function PageEdit(props) {
      var attributes = props.attributes;

      return (
        wp.element.createElement(
          wp.element.Fragment,
          null,
          wp.element.createElement(
            InspectorControls,
            null,
            wp.element.createElement(
              PanelBody,
              {
                title: __('Page Embed Settings', 'instantpay'),
                initialOpen: true,
              },
              wp.element.createElement(TextControl, {
                label: __('Username (optional)', 'instantpay'),
                help: __(
                  'Leave empty to use the InstantPay account connected in the InstantPay admin page.',
                  'instantpay'
                ),
                placeholder: 'yourusername',
                value: attributes.username || '',
                onChange: function (value) {
                  props.setAttributes({ username: value });
                },
              })
            )
          ),
          wp.element.createElement(
            'div',
            {
              className:
                'instantpay-block-preview instantpay-block-preview-page',
              style: {
                border: '1px dashed #e5e7eb',
                padding: '12px 14px',
                borderRadius: '8px',
                background: '#ffffff',
              },
            },
            wp.element.createElement(
              'strong',
              null,
              __('InstantPay Payment Page', 'instantpay')
            ),
            wp.element.createElement('br', null),
            wp.element.createElement(
              'span',
              { style: { fontSize: '12px', color: '#6b7280' } },
              attributes.username
                ? __('Username override:', 'instantpay') +
                    ' ' +
                    attributes.username
                : __('Using connected InstantPay account', 'instantpay')
            )
          )
        )
      );
    },
    save: function () {
      // Dynamic block – rendered via PHP shortcode on the frontend.
      return null;
    },
  });
})();


