/**
 * InstantPay frontend embed helper.
 *
 * Listens for postMessage events from https://instantpay.now iframes
 * and adjusts their height to avoid inner scrollbars.
 */

(function () {
  if (typeof window === 'undefined') return;

  var INSTANTPAY_ORIGIN = 'https://instantpay.now';

  function isInstantPayOrigin(origin) {
    return origin === INSTANTPAY_ORIGIN;
  }

  function updateIframeHeight(message) {
    if (!message || typeof message.height !== 'number') return;

    var iframes = document.querySelectorAll('iframe[src^="' + INSTANTPAY_ORIGIN + '"]');
    if (!iframes.length) return;

    var targetId = message.iframeId || message.id || null;

    // Generous buffer so widget border and shadow are never clipped,
    // even with different themes/fonts around the iframe.
    var finalHeight = message.height + 80;

    if (targetId) {
      for (var i = 0; i < iframes.length; i++) {
        var frame = iframes[i];
        if (
          frame.getAttribute('data-instantpay-id') === String(targetId) ||
          frame.id === String(targetId)
        ) {
          frame.style.height = finalHeight + 'px';
          return;
        }
      }
    }

    // Fallback: set height on all InstantPay iframes on the page.
    for (var j = 0; j < iframes.length; j++) {
      iframes[j].style.height = finalHeight + 'px';
    }
  }

  window.addEventListener(
    'message',
    function (event) {
      if (!isInstantPayOrigin(event.origin)) return;

      var data = event.data || {};

      // Support both generic { height } and namespaced payloads.
      if (typeof data.height === 'number') {
        updateIframeHeight(data);
        return;
      }

      if (data.type === 'instantpay-embed-resize' && data.payload) {
        updateIframeHeight(data.payload);
        return;
      }

      // Backwards compatibility: widget'lar şu an { type: 'iframe-resize', height } formatında gönderiyor.
      if (data.type === 'iframe-resize' && typeof data.height === 'number') {
        updateIframeHeight(data);
      }
    },
    false
  );
})();


