<?php
/**
 * Admin page for InstantPay – Solana Payments for WordPress.
 *
 * @package InstantPay
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class InstantPay_Admin_Page {

	/**
	 * Register admin menu.
	 *
	 * @return void
	 */
	public function register_menu() {
		add_menu_page(
			__( 'InstantPay', 'instantpay-solana-payments' ),
			__( 'InstantPay', 'instantpay-solana-payments' ),
			'manage_options',
			'instantpay',
			array( $this, 'render_page' ),
			'dashicons-cart',
			56
		);
	}

	/**
	 * Render main admin page with embedded dashboard iframe.
	 *
	 * @return void
	 */
	public function render_page() {
		$options   = get_option( 'instantpay_options', array() );
		$username  = isset( $options['username'] ) ? sanitize_text_field( $options['username'] ) : '';
		$site_token = isset( $options['site_token'] ) ? sanitize_text_field( $options['site_token'] ) : '';

		$site_url = home_url();

		// Build iframe src URL.
		$base_url = 'https://instantpay.now/wp-embedded';
		$args     = array(
			'site_url' => $site_url,
		);

		if ( ! empty( $username ) ) {
			$args['username'] = $username;
		}

		if ( ! empty( $site_token ) ) {
			$args['token'] = $site_token;
		}

		$iframe_src = esc_url( add_query_arg( $args, $base_url ) );

		?>
		<div class="wrap instantpay-admin-wrap">
			<div class="instantpay-header-card">
				<div class="instantpay-header-logo">
					<img src="<?php echo esc_url( INSTANTPAY_PLUGIN_URL . 'assets/images/web_logo.png' ); ?>" alt="<?php esc_attr_e( 'InstantPay Logo', 'instantpay-solana-payments' ); ?>" class="instantpay-logo-img" />
				</div>
				<div class="instantpay-header-text">
					<h1><?php esc_html_e( 'InstantPay – Solana Payments', 'instantpay-solana-payments' ); ?></h1>
					<p class="instantpay-header-subtitle">
						<?php esc_html_e( 'Connect your wallet in the embedded dashboard below and manage your InstantPay account without leaving WordPress.', 'instantpay-solana-payments' ); ?>
					</p>
				</div>
				<div class="instantpay-status">
					<?php 
					// Username set edilmişse (boş string olsa bile) bağlı sayılır
					// Boş string = cüzdan bağlı ama username henüz seçilmemiş
					// null/undefined = cüzdan bağlı değil
					$is_connected = isset( $options['username'] );
					?>
					<?php if ( $is_connected ) : ?>
						<p class="instantpay-status-connected">
							<span class="instantpay-status-dot instantpay-status-dot--connected"></span>
							<?php if ( ! empty( $username ) ) : ?>
								<?php
								printf(
									/* translators: %s: InstantPay username */
									esc_html__( 'Connected as %s', 'instantpay-solana-payments' ),
									'<strong>' . esc_html( $username ) . '</strong>'
								);
								?>
							<?php else : ?>
								<?php esc_html_e( 'Wallet connected. Please select a username in the dashboard below.', 'instantpay-solana-payments' ); ?>
							<?php endif; ?>
						</p>
					<?php else : ?>
						<p class="instantpay-status-disconnected">
							<span class="instantpay-status-dot instantpay-status-dot--disconnected"></span>
							<?php esc_html_e( 'Not connected yet. Use the embedded dashboard below to connect your wallet.', 'instantpay-solana-payments' ); ?>
						</p>
					<?php endif; ?>
				</div>
				<div class="instantpay-header-actions">
					<button type="button" class="button instantpay-refresh-button" id="instantpay-refresh-button">
						<?php esc_html_e( 'Refresh status', 'instantpay-solana-payments' ); ?>
					</button>
					<p class="instantpay-header-helper">
						<?php esc_html_e( 'After connecting or disconnecting in the dashboard below, click Refresh status to update this panel.', 'instantpay-solana-payments' ); ?>
					</p>
				</div>
				<div class="instantpay-howto-section">
					<button type="button" class="instantpay-howto-toggle" id="instantpay-howto-toggle" aria-expanded="false">
						<span class="instantpay-howto-title"><?php esc_html_e( 'How to use', 'instantpay-solana-payments' ); ?></span>
						<span class="instantpay-howto-arrow">▼</span>
					</button>
					<div class="instantpay-howto-content" id="instantpay-howto-content">
						<p class="instantpay-howto-intro">
							<?php esc_html_e( 'After connecting your wallet and InstantPay account below, you can add payment widgets to your content in two ways:', 'instantpay-solana-payments' ); ?>
						</p>
						<ul class="instantpay-howto-list">
							<li>
								<strong><?php esc_html_e( 'Gutenberg blocks:', 'instantpay-solana-payments' ); ?></strong>
								<?php esc_html_e( 'In the block editor, search for "InstantPay" to insert Donate, Product or Page blocks and configure them from the right-hand sidebar.', 'instantpay-solana-payments' ); ?>
							</li>
							<li>
								<strong><?php esc_html_e( 'Shortcodes:', 'instantpay-solana-payments' ); ?></strong>
								<code>[instantpay_donate amount="0.5" theme="dark"]</code>,
								<code>[instantpay_product id="YOUR_PRODUCT_ID"]</code>,
								<code>[instantpay_page]</code>
							</li>
						</ul>
					</div>
				</div>
			</div>

			<div class="instantpay-admin-iframe-container">
				<iframe
					id="instantpay-admin-iframe"
					src="<?php echo $iframe_src; ?>"
					title="<?php esc_attr_e( 'InstantPay Dashboard', 'instantpay-solana-payments' ); ?>"
					style="width: 100%; min-height: 700px; border: 1px solid #ddd; border-radius: 8px; background: #fff;"
				></iframe>
			</div>

			<?php
			// Enqueue admin iframe script.
			wp_enqueue_script(
				'instantpay-admin-iframe',
				INSTANTPAY_PLUGIN_URL . 'assets/js/admin-iframe.js',
				array( 'jquery' ),
				INSTANTPAY_PLUGIN_VERSION,
				true
			);

			wp_localize_script(
				'instantpay-admin-iframe',
				'InstantPayAdmin',
				array(
					'iframeId'      => 'instantpay-admin-iframe',
					// WordPress tarafında çoğunlukla www'li domain kullanıldığı için
					// burada production origin'i açıkça www ile tanımlıyoruz.
					'allowedOrigin' => 'https://www.instantpay.now',
					'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
					// CSRF koruması için nonce
					'nonce'         => wp_create_nonce( 'instantpay_update_connection' ),
				)
			);
			?>
		</div>
		<?php
		// Enqueue admin styles at the end to ensure they are loaded for this page.
		wp_enqueue_style(
			'instantpay-admin',
			INSTANTPAY_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			INSTANTPAY_PLUGIN_VERSION
		);
		?>
		<?php
	}
}


