<?php
/**
 * Main plugin class for InstantPay – Solana Payments for WordPress.
 *
 * @package InstantPay
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class InstantPay_Plugin {

	/**
	 * Run the plugin.
	 *
	 * @return void
	 */
	public function run() {
		$this->load_textdomain();
		$this->define_hooks();
	}

	/**
	 * Load plugin textdomain for translations.
	 * Note: When hosted on WordPress.org, translations are loaded automatically.
	 * This method is kept for backward compatibility but does nothing.
	 *
	 * @return void
	 */
	public function load_textdomain() {
		// WordPress.org automatically loads translations for plugins hosted there.
		// This method is kept for backward compatibility but is no longer needed.
		// Text domain: instantpay-solana-payments (must match plugin slug)
	}

	/**
	 * Register hooks for admin and frontend.
	 *
	 * @return void
	 */
	private function define_hooks() {
		// Admin page.
		require_once INSTANTPAY_PLUGIN_DIR . 'includes/class-instantpay-admin-page.php';
		$admin_page = new InstantPay_Admin_Page();
		add_action( 'admin_menu', array( $admin_page, 'register_menu' ) );

		// Admin AJAX.
		require_once INSTANTPAY_PLUGIN_DIR . 'includes/class-instantpay-admin-ajax.php';
		$admin_ajax = new InstantPay_Admin_Ajax();
		add_action( 'wp_ajax_instantpay_update_connection', array( $admin_ajax, 'handle_update_connection' ) );

		// Settings.
		require_once INSTANTPAY_PLUGIN_DIR . 'includes/class-instantpay-settings.php';
		$settings = new InstantPay_Settings();
		add_action( 'admin_init', array( $settings, 'register_settings' ) );

		// Shortcodes.
		require_once INSTANTPAY_PLUGIN_DIR . 'includes/class-instantpay-shortcodes.php';
		$shortcodes = new InstantPay_Shortcodes();
		add_action( 'init', array( $shortcodes, 'register_shortcodes' ) );

		// Blocks (v0: basic registration).
		require_once INSTANTPAY_PLUGIN_DIR . 'includes/class-instantpay-blocks.php';
		$blocks = new InstantPay_Blocks();
		add_action( 'init', array( $blocks, 'register_blocks' ) );

		// Block editor assets (Gutenberg).
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );

		// Frontend assets.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );

		// Admin assets (for menu styling).
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
	}

	/**
	 * Enqueue admin assets (CSS for menu styling).
	 *
	 * @return void
	 */
	public function enqueue_admin_assets() {
		wp_enqueue_style(
			'instantpay-admin',
			INSTANTPAY_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			INSTANTPAY_PLUGIN_VERSION
		);
	}

	/**
	 * Enqueue frontend scripts for embed handling.
	 *
	 * @return void
	 */
	public function enqueue_frontend_assets() {
		$options            = get_option( 'instantpay_options', array() );
		$auto_append_script = isset( $options['auto_append_script'] ) ? (int) $options['auto_append_script'] : 1;

		if ( $auto_append_script ) {
			wp_enqueue_script(
				'instantpay-frontend-embed',
				INSTANTPAY_PLUGIN_URL . 'assets/js/frontend-embed.js',
				array(),
				INSTANTPAY_PLUGIN_VERSION,
				true
			);
		}
	}

	/**
	 * Enqueue block editor assets.
	 *
	 * @return void
	 */
	public function enqueue_block_editor_assets() {
		wp_enqueue_script(
			'instantpay-blocks',
			INSTANTPAY_PLUGIN_URL . 'assets/js/blocks.js',
			array( 'wp-blocks', 'wp-element', 'wp-components', 'wp-i18n', 'wp-editor', 'wp-block-editor' ),
			INSTANTPAY_PLUGIN_VERSION,
			true
		);

		// Pass basic config to the block editor script.
		wp_localize_script(
			'instantpay-blocks',
			'InstantPayBlocks',
			array(
				// Base URL of the InstantPay app for public API calls.
				// Not: Vercel tarafında bare domain -> www yönlendirmesi olduğu için
				// CORS sorunları yaşamamak adına burada doğrudan www'li domain kullanıyoruz.
				'apiBase'  => 'https://www.instantpay.now',
				// Currently connected username (if any).
				'username' => self::get_username(),
			)
		);
	}

	/**
	 * Helper: get stored username.
	 *
	 * @return string
	 */
	public static function get_username() {
		$options  = get_option( 'instantpay_options', array() );
		$username = isset( $options['username'] ) ? $options['username'] : '';

		return is_string( $username ) ? $username : '';
	}

	/**
	 * Helper: get stored site token.
	 *
	 * @return string
	 */
	public static function get_site_token() {
		$options   = get_option( 'instantpay_options', array() );
		$site_token = isset( $options['site_token'] ) ? $options['site_token'] : '';

		return is_string( $site_token ) ? $site_token : '';
	}
}


