<?php
/**
 * Settings registration for InstantPay – Solana Payments for WordPress.
 *
 * @package InstantPay
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class InstantPay_Settings {

	/**
	 * Register settings.
	 *
	 * @return void
	 */
	public function register_settings() {
		register_setting(
			'instantpay_options_group',
			'instantpay_options',
			array( $this, 'sanitize_options' )
		);

		add_settings_section(
			'instantpay_main_section',
			__( 'InstantPay Settings', 'instantpay-solana-payments' ),
			array( $this, 'render_main_section' ),
			'instantpay-solana-payments'
		);

		add_settings_field(
			'username',
			__( 'InstantPay Username', 'instantpay-solana-payments' ),
			array( $this, 'render_username_field' ),
			'instantpay-solana-payments',
			'instantpay_main_section'
		);

		add_settings_field(
			'site_token',
			__( 'Site Token', 'instantpay-solana-payments' ),
			array( $this, 'render_site_token_field' ),
			'instantpay-solana-payments',
			'instantpay_main_section'
		);

		add_settings_field(
			'auto_append_script',
			__( 'Auto-append Embed Script', 'instantpay-solana-payments' ),
			array( $this, 'render_auto_append_field' ),
			'instantpay-solana-payments',
			'instantpay_main_section'
		);
	}

	/**
	 * Sanitize options callback.
	 *
	 * @param array $input Raw input.
	 * @return array
	 */
	public function sanitize_options( $input ) {
		$output = array();

		if ( isset( $input['username'] ) ) {
			$output['username'] = sanitize_text_field( $input['username'] );
		}

		if ( isset( $input['site_token'] ) ) {
			$output['site_token'] = sanitize_text_field( $input['site_token'] );
		}

		$output['auto_append_script'] = isset( $input['auto_append_script'] ) ? (int) $input['auto_append_script'] : 0;

		return $output;
	}

	/**
	 * Section description.
	 *
	 * @return void
	 */
	public function render_main_section() {
		echo '<p>' . esc_html__( 'Basic settings for your InstantPay integration.', 'instantpay-solana-payments' ) . '</p>';
	}

	/**
	 * Username field.
	 *
	 * @return void
	 */
	public function render_username_field() {
		$options  = get_option( 'instantpay_options', array() );
		$username = isset( $options['username'] ) ? $options['username'] : '';

		printf(
			'<input type="text" id="instantpay_username" name="instantpay_options[username]" value="%s" class="regular-text" />',
			esc_attr( $username )
		);
		echo '<p class="description">' . esc_html__( 'Your InstantPay username (used for /u/username and default widget embeds).', 'instantpay-solana-payments' ) . '</p>';
	}

	/**
	 * Site token field.
	 *
	 * @return void
	 */
	public function render_site_token_field() {
		$options   = get_option( 'instantpay_options', array() );
		$site_token = isset( $options['site_token'] ) ? $options['site_token'] : '';

		printf(
			'<input type="text" id="instantpay_site_token" name="instantpay_options[site_token]" value="%s" class="regular-text" />',
			esc_attr( $site_token )
		);
		echo '<p class="description">' . esc_html__( 'Optional token that links this WordPress site to your InstantPay account (managed via the embedded dashboard).', 'instantpay-solana-payments' ) . '</p>';
	}

	/**
	 * Auto append script field.
	 *
	 * @return void
	 */
	public function render_auto_append_field() {
		$options            = get_option( 'instantpay_options', array() );
		$auto_append_script = isset( $options['auto_append_script'] ) ? (int) $options['auto_append_script'] : 1;

		printf(
			'<label><input type="checkbox" name="instantpay_options[auto_append_script]" value="1" %s /> %s</label>',
			checked( 1, $auto_append_script, false ),
			esc_html__( 'Automatically load the InstantPay embed script on pages where shortcodes/blocks are used.', 'instantpay-solana-payments' )
		);
	}
}


