<?php
/**
 * Shortcodes for InstantPay – Solana Payments for WordPress.
 *
 * @package InstantPay
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class InstantPay_Shortcodes {

	/**
	 * Register shortcodes.
	 *
	 * @return void
	 */
	public function register_shortcodes() {
		add_shortcode( 'instantpay_donate', array( $this, 'render_donate_shortcode' ) );
		add_shortcode( 'instantpay_product', array( $this, 'render_product_shortcode' ) );
		add_shortcode( 'instantpay_page', array( $this, 'render_page_shortcode' ) );
	}

	/**
	 * Donate widget shortcode.
	 *
	 * [instantpay_donate username="john" amount="0.5" theme="light"]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function render_donate_shortcode( $atts ) {
		$atts = shortcode_atts(
			array(
				'username' => InstantPay_Plugin::get_username(),
				'amount'   => '',
				'theme'    => 'light',
			),
			$atts,
			'instantpay_donate'
		);

		$username = sanitize_text_field( $atts['username'] );
		$amount   = sanitize_text_field( $atts['amount'] );
		$theme    = sanitize_text_field( $atts['theme'] );

		if ( empty( $username ) ) {
			return '<p>' . esc_html__( 'InstantPay is not fully configured yet. Please connect your account in the InstantPay settings page.', 'instantpay-solana-payments' ) . '</p>';
		}

		$query_args = array();
		if ( $amount !== '' ) {
			$query_args['amount'] = $amount;
		}
		if ( $theme !== '' ) {
			$query_args['theme'] = $theme;
		}

		$src = 'https://instantpay.now/embed/' . rawurlencode( $username ) . '/donate';
		if ( ! empty( $query_args ) ) {
			$src = add_query_arg( $query_args, $src );
		}

		$src = esc_url( $src );

		ob_start();
		?>
		<!-- ***INSTANTPAY PAYMENT SYSTEM*** -->
		<div class="instantpay-embed-container instantpay-embed-donate">
		<iframe
			src="<?php echo esc_url( $src ); ?>"
			title="<?php esc_attr_e( 'InstantPay Donate Widget', 'instantpay-solana-payments' ); ?>"
			width="100%"
			height="550"
			style="width:100%;height:550px;border:0;overflow:hidden;"
			scrolling="no"
			data-instantpay-embed="donate"
		></iframe>
		</div>
		<!-- ***INSTANTPAY PAYMENT SYSTEM*** -->
		<?php
		return ob_get_clean();
	}

	/**
	 * Product widget shortcode.
	 *
	 * [instantpay_product id="product-id" username="john" theme="light"]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function render_product_shortcode( $atts ) {
		$atts = shortcode_atts(
			array(
				'id'       => '',
				'username' => InstantPay_Plugin::get_username(),
				'theme'    => 'light',
			),
			$atts,
			'instantpay_product'
		);

		$product_id = sanitize_text_field( $atts['id'] );
		$username   = sanitize_text_field( $atts['username'] );
		$theme      = sanitize_text_field( $atts['theme'] );

		if ( empty( $product_id ) ) {
			return '<p>' . esc_html__( 'No InstantPay product selected. Please add a product ID.', 'instantpay-solana-payments' ) . '</p>';
		}

		if ( empty( $username ) ) {
			return '<p>' . esc_html__( 'InstantPay is not fully configured yet. Please connect your account in the InstantPay settings page.', 'instantpay-solana-payments' ) . '</p>';
		}

		$query_args = array();
		if ( $theme !== '' ) {
			$query_args['theme'] = $theme;
		}

		$src = 'https://instantpay.now/embed/' . rawurlencode( $username ) . '/buy/' . rawurlencode( $product_id );
		if ( ! empty( $query_args ) ) {
			$src = add_query_arg( $query_args, $src );
		}

		$src = esc_url( $src );

		ob_start();
		?>
		<!-- ***INSTANTPAY PAYMENT SYSTEM*** -->
		<div class="instantpay-embed-container instantpay-embed-product">
		<iframe
			src="<?php echo esc_url( $src ); ?>"
			title="<?php esc_attr_e( 'InstantPay Product Widget', 'instantpay-solana-payments' ); ?>"
			width="100%"
			height="260"
			style="width:100%;height:260px;border:0;overflow:hidden;"
			scrolling="no"
			data-instantpay-embed="product"
		></iframe>
		</div>
		<!-- ***INSTANTPAY PAYMENT SYSTEM*** -->
		<?php
		return ob_get_clean();
	}

	/**
	 * Full payment page embed shortcode.
	 *
	 * [instantpay_page username="john"]
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function render_page_shortcode( $atts ) {
		$atts = shortcode_atts(
			array(
				'username' => InstantPay_Plugin::get_username(),
			),
			$atts,
			'instantpay_page'
		);

		$username = sanitize_text_field( $atts['username'] );

		if ( empty( $username ) ) {
			return '<p>' . esc_html__( 'InstantPay is not fully configured yet. Please connect your account in the InstantPay settings page.', 'instantpay-solana-payments' ) . '</p>';
		}

		$src = 'https://instantpay.now/u/' . rawurlencode( $username );
		$src = esc_url( $src );

		ob_start();
		?>
		<!-- ***INSTANTPAY PAYMENT SYSTEM*** -->
		<div class="instantpay-embed-container instantpay-embed-page">
		<iframe
			src="<?php echo esc_url( $src ); ?>"
			title="<?php esc_attr_e( 'InstantPay Payment Page', 'instantpay-solana-payments' ); ?>"
			width="100%"
			height="1500"
			style="width:100%;height:1500px;border:0;overflow:auto;"
			scrolling="yes"
			data-instantpay-embed="page"
		></iframe>
		</div>
		<!-- ***INSTANTPAY PAYMENT SYSTEM*** -->
		<?php
		return ob_get_clean();
	}
}


